///////////////////////////////////////////////////////////////////////////////
// gradient.h
// ==========
// 1st order derivative for horizontal and vertical direction.
// It computes horizontal and vertical gradient vectors of an image using 
// central difference.
// deltaX(x,y) = I(x+1, y) - I(x-1, y)
// deltaY(x,y) = I(x, y+1) - I(x, y-1)
//
// The gradients can be negative values, so the gradient output must be a 
// signed array. For example, for 8-bit char output, the range of values are
// from -128 to +127. If you want to store the results into "unsigned char", 
// use gradientNorm() instead. All values are shifted by 128 in gradientNorm().
// As a result, the normalized gradient image looks like with emboss effect.
//
// Mostly, the magnitude of the gragients are useful for image processing. It 
// is computed using;
// Mag(x,y) = sqrt(deltaX(x,y)*deltaX(x,y) + deltaY(x,y)*deltaY(x,y))
//
// Or, we can approximates by adding absolute values.
// Mag(x,y) ~ abs(deltaX(x,y)) + abs(deltaY(x,y))
//
//  AUTHOR: Song Ho Ahn (song.ahn@gmail.com)
// CREATED: 2006-10-30
// UPDATED: 2006-10-30
///////////////////////////////////////////////////////////////////////////////

#ifndef DIP_GRADIENT_H
#define DIP_GRADIENT_H

namespace dip
{
    ///////////////////////////////////////////////////////////////////////////
    // compute gradient (first order derivative x and y) using central difference
    // delta(t) = f(t+1) - f(t-1)
    // The pointer to magnitude array is optional.
    ///////////////////////////////////////////////////////////////////////////
    bool gradient(const unsigned char* src, int width, int height, int* deltaX, int* deltaY, int* mag=0);
    bool gradient(const unsigned char* src, int width, int height, char* deltaX, char* deltaY, unsigned char* mag=0);
    bool gradientNorm(const unsigned char* src, int width, int height, unsigned char* deltaX, unsigned char* deltaY, unsigned char* mag=0);



    ///////////////////////////////////////////////////////////////////////////////
    // compute magnitude of gradient(deltaX & deltaY) per pixel
    ///////////////////////////////////////////////////////////////////////////////
    bool gradientMagnitude(const int* deltaX, const int* deltaY, int width, int height, int* mag);

}

#endif
