///////////////////////////////////////////////////////////////////////////////
// main.cpp
// ========
// testing GL_ARB_vertex_array_object extension with VBO
// It draws a cube and tetrahedron using glDrawElements() and glVertexArrays()
//
// OpenGL Spec v3.3 Core Profile. Appendix E.2.2 Removed Features (p.344)
// Client vertex and index arrays - all vertex array attribute and element array
// index pointers must refer to buffer objects. The default vertex array object
// (the name zero) is also deprecated. Calling VertexAttribPointer when no
// buffer object or no vertex array object is bound will generate an INVALID_-
// OPERATION error, as will calling any array drawing command when no ver-
// tex array object is bound.

//  AUTHOR: Song Ho Ahn (song.ahn@gmail.com)
// CREATED: 2012-07-09
// UPDATED: 2023-05-14
///////////////////////////////////////////////////////////////////////////////

#include <glad/glad.h>
#include <GLFW/glfw3.h>

#include <cstdlib>
#include <iostream>
#include <sstream>
#include <iomanip>
#include "BitmapFontData.h"     // to draw bitmap font with GLFW
#include "fontCourier20.h"      // font:courier new, height:20px
#include "Matrices.h"
#include "Bmp.h"

// glfw callbacks
void errorCallback(int error, const char* description);
void framebufferSizeCallback(GLFWwindow* window, int width, int height);
void keyCallback(GLFWwindow* window, int key, int scancode, int action, int mods);
void mouseButtonCallback(GLFWwindow* window, int button, int action, int mods);
void cursorPosCallback(GLFWwindow* window, double x, double y);

// function prototypes
void preFrame(double frameTime);
void frame(double frameTime);
void postFrame(double frameTime);
void toPerspective();
void toOrtho();
void initGL();
bool initGLSL();
void initVBO();
bool initSharedMem();
void clearSharedMem();
void showInfo();


// Bui-Tuong Phong shading model with color + texture =========================
const char* vsSource = R"(
// GLSL version (OpenGL 3.3)
#version 330
// uniforms
uniform mat4 matrixModelView;
uniform mat4 matrixNormal;
uniform mat4 matrixModelViewProjection;
// vertex attribs (input)
layout(location=0) in vec3 vertexPosition;
layout(location=1) in vec3 vertexNormal;
layout(location=2) in vec3 vertexColor;
layout(location=3) in vec2 vertexTexCoord;
// varyings (output)
out vec3 esVertex;
out vec3 esNormal;
out vec4 color;
out vec2 texCoord0;
void main()
{
    esVertex = vec3(matrixModelView * vec4(vertexPosition, 1.0));
    esNormal = vec3(matrixNormal * vec4(vertexNormal, 1.0));
    color = vec4(vertexColor, 1.0);
    texCoord0 = vertexTexCoord;
    gl_Position = matrixModelViewProjection * vec4(vertexPosition, 1.0);
}
)";

const char* fsSource = R"(
// GLSL version (OpenGL 3.3)
#version 330
// uniforms
uniform vec4 lightPosition;             // should be in the eye space
uniform vec4 lightAmbient;              // light ambient color
uniform vec4 lightDiffuse;              // light diffuse color
uniform vec4 lightSpecular;             // light specular color
uniform sampler2D map0;                 // texture map #1
uniform bool textureUsed;               // flag for texture
// varyings (input)
in vec3 esVertex;
in vec3 esNormal;
in vec4 color;
in vec2 texCoord0;
// output
out vec4 fragColor;
void main()
{
    vec3 normal = normalize(esNormal);
    vec3 light;
    if(lightPosition.w == 0.0)
    {
        light = normalize(lightPosition.xyz);
    }
    else
    {
        light = normalize(lightPosition.xyz - esVertex);
    }
    vec3 view = normalize(-esVertex);
    vec3 reflectVec = reflect(-light, normal);  // 2 * N * (N dot L) - L

    fragColor = lightAmbient * color;                       // begin with ambient
    float dotNL = max(dot(normal, light), 0.0);
    fragColor += lightDiffuse * color * dotNL;              // add diffuse
    if(textureUsed)
        fragColor *= texture(map0, texCoord0);              // modulate texture map
    float dotVR = max(dot(view, reflectVec), 0.0);
    fragColor += pow(dotVR, 128.0) * lightSpecular * color; // add specular
    fragColor.a = color.a;                                  // keep alpha as diffuse
}
)";



// constants
const int   WINDOW_WIDTH    = 1000;
const int   WINDOW_HEIGHT   = 600;
const float CAMERA_DISTANCE = 6.0f;


// global variables
GLFWwindow* window;
int windowWidth;
int windowHeight;
int fbWidth;
int fbHeight;
double runTime;
bool mouseLeftDown;
bool mouseRightDown;
bool mouseMiddleDown;
double mouseX, mouseY;
float cameraAngleX;
float cameraAngleY;
float cameraDistance;
int drawMode;
GLuint vboCube;                     // VBO ID for cube
GLuint vboTetra;                    // VBO ID for tetrahedron
GLuint iboCube;                     // VBO ID for cube index
GLuint vaoDefault;                  // VAO ID for default
GLuint vaoCube;                     // VAO ID for cube
GLuint vaoTetra;                    // VAO ID for tetrahedron
GLuint texId = 0;                   // ID of texture object
BitmapFontData bmFont;
Matrix4 matrixModelView;
Matrix4 matrixProjection;
// GLSL
GLuint progId = 0;                  // ID of GLSL program
GLint uniformMatrixModelView;
GLint uniformMatrixModelViewProjection;
GLint uniformMatrixNormal;
GLint uniformLightPosition;
GLint uniformLightAmbient;
GLint uniformLightDiffuse;
GLint uniformLightSpecular;
GLint uniformMap0;
GLint uniformTextureUsed;
GLint attribVertexPosition;     // 0
GLint attribVertexNormal;       // 1
GLint attribVertexColor;        // 2
GLint attribVertexTexCoord;     // 3



// unit cube //////////////////////////////////////////////////////////////////
//    v6----- v5
//   /|      /|
//  v1------v0|
//  | |     | |
//  | v7----|-v4
//  |/      |/
//  v2------v3

// vertex coords array for glDrawElements() ===================================
// A cube has 6 sides and each side has 4 vertices, therefore, the total number
// of vertices is 24 (6 sides * 4 verts), and 72 floats in the vertex array
// since each vertex has 3 components (x,y,z) (= 24 * 3)
GLfloat vertices1[] = {
    1, 1, 1,  -1, 1, 1,  -1,-1, 1,   1,-1, 1,   // v0,v1,v2,v3 (front)
    1, 1, 1,   1,-1, 1,   1,-1,-1,   1, 1,-1,   // v0,v3,v4,v5 (right)
    1, 1, 1,   1, 1,-1,  -1, 1,-1,  -1, 1, 1,   // v0,v5,v6,v1 (top)
   -1, 1, 1,  -1, 1,-1,  -1,-1,-1,  -1,-1, 1,   // v1,v6,v7,v2 (left)
   -1,-1,-1,   1,-1,-1,   1,-1, 1,  -1,-1, 1,   // v7,v4,v3,v2 (bottom)
    1,-1,-1,  -1,-1,-1,  -1, 1,-1,   1, 1,-1    // v4,v7,v6,v5 (back)
};

// normal array
GLfloat normals1[]  = {
    0, 0, 1,   0, 0, 1,   0, 0, 1,   0, 0, 1,   // v0,v1,v2,v3 (front)
    1, 0, 0,   1, 0, 0,   1, 0, 0,   1, 0, 0,   // v0,v3,v4,v5 (right)
    0, 1, 0,   0, 1, 0,   0, 1, 0,   0, 1, 0,   // v0,v5,v6,v1 (top)
   -1, 0, 0,  -1, 0, 0,  -1, 0, 0,  -1, 0, 0,   // v1,v6,v7,v2 (left)
    0,-1, 0,   0,-1, 0,   0,-1, 0,   0,-1, 0,   // v7,v4,v3,v2 (bottom)
    0, 0,-1,   0, 0,-1,   0, 0,-1,   0, 0,-1    // v4,v7,v6,v5 (back)
};

// color array
GLfloat colors1[]   = {
    1, 1, 1,   0, 1, 1,   0, 0, 1,   1, 0, 1,   // v0,v1,v2,v3 (front)
    1, 1, 1,   1, 0, 1,   1, 0, 0,   1, 1, 0,   // v0,v3,v4,v5 (right)
    1, 1, 1,   1, 1, 0,   0, 1, 0,   0, 1, 1,   // v0,v5,v6,v1 (top)
    0, 1, 1,   0, 1, 0,   0, 0, 0,   0, 0, 1,   // v1,v6,v7,v2 (left)
    0, 0, 0,   1, 0, 0,   1, 0, 1,   0, 0, 1,   // v7,v4,v3,v2 (bottom)
    1, 0, 0,   0, 0, 0,   0, 1, 0,   1, 1, 0    // v4,v7,v6,v5 (back)
};

// texCoord array
GLfloat texCoords1[] = {
    1, 0,   0, 0,   0, 1,   1, 1,               // v0,v1,v2,v3 (front)
    0, 0,   0, 1,   1, 1,   1, 0,               // v0,v3,v4,v5 (right)
    1, 1,   1, 0,   0, 0,   0, 1,               // v0,v5,v6,v1 (top)
    1, 0,   0, 0,   0, 1,   1, 1,               // v1,v6,v7,v2 (left)
    0, 1,   1, 1,   1, 0,   0, 0,               // v7,v4,v3,v2 (bottom)
    0, 1,   1, 1,   1, 0,   0, 0                // v4,v7,v6,v5 (back)
};

// index array for glDrawElements() ===========================================
// A cube has 36 indices = 6 sides * 2 tris * 3 verts
GLuint indices1[] = {
     0, 1, 2,   2, 3, 0,    // v0-v1-v2, v2-v3-v0 (front)
     4, 5, 6,   6, 7, 4,    // v0-v3-v4, v4-v5-v0 (right)
     8, 9,10,  10,11, 8,    // v0-v5-v6, v6-v1-v0 (top)
    12,13,14,  14,15,12,    // v1-v6-v7, v7-v2-v1 (left)
    16,17,18,  18,19,16,    // v7-v4-v3, v3-v2-v7 (bottom)
    20,21,22,  22,23,20     // v4-v7-v6, v6-v5-v4 (back)
};


// vertex arrays for tetrahedron ==============================================
const float N = 0.57735f;   // normal component
GLfloat vertices2[] = {
    1, 1, 1,  -1, 1,-1,  -1,-1, 1,      // face 1
    1, 1, 1,  -1,-1, 1,   1,-1,-1,      // face 2
    1, 1, 1,   1,-1,-1,  -1, 1,-1,      // face 3
   -1,-1, 1,  -1, 1,-1,   1,-1,-1       // face 4
};

GLfloat normals2[]  = {
   -N, N, N,  -N, N, N,  -N, N, N,      // face 1
    N,-N, N,   N,-N, N,   N,-N, N,      // face 2
    N, N,-N,   N, N,-N,   N, N,-N,      // face 3
   -N,-N,-N,  -N,-N,-N,  -N,-N,-N       // face 4
};

GLfloat colors2[]   = {
    1, 1, 1,   0, 1, 0,   0, 0, 1,      // face 1
    1, 1, 1,   0, 0, 1,   1, 0, 0,      // face 2
    1, 1, 1,   1, 0, 0,   0, 1, 0,      // face 3
    0, 0, 1,   0, 1, 0,   1, 0, 0       // face 4
};



///////////////////////////////////////////////////////////////////////////////
int main(int argc, char **argv)
{
    // init global vars
    initSharedMem();

    // init GLFW
    glfwInit();
    glfwWindowHint(GLFW_CONTEXT_VERSION_MAJOR, 3);
    glfwWindowHint(GLFW_CONTEXT_VERSION_MINOR, 3);
    glfwWindowHint(GLFW_OPENGL_PROFILE, GLFW_OPENGL_CORE_PROFILE);
    //glfwWindowHint(GLFW_OPENGL_PROFILE, GLFW_OPENGL_COMPAT_PROFILE);
    //NOTE: mac only supports forward-compatible, core profile for v3.x & v4.x
    #ifdef __APPLE__
        glfwWindowHint(GLFW_OPENGL_FORWARD_COMPAT, GL_TRUE);
    #endif

    window = glfwCreateWindow(WINDOW_WIDTH, WINDOW_HEIGHT, argv[0], 0, 0);
    if(window)
    {
        int ww, wh, fw, fh;
        glfwGetWindowSize(window, &ww, &wh);
        glfwGetFramebufferSize(window, &fw, &fh);
        std::cout << "GLFW window is created: " << ww << "x" << wh
                  << " (Framebuffer: " << fw << "x" << fh << ")" << std::endl;
    }
    else
    {
        std::cout << "[ERROR] Failed to create GLFW window" << std::endl;
        glfwTerminate();
        exit(1);
    }

    // set OpenGL RC
    glfwMakeContextCurrent(window);

    // init glad to load gl extensions
    gladLoadGLLoader((GLADloadproc)glfwGetProcAddress);

    //DEBUG: print glfw/opengl info
    //int majorVersion = 0, minorVersion = 0, revision = 0;
    //glfwGetVersion(&majorVersion, &minorVersion, &revision);
    //std::cout << "OpenGL Version: " << majorVersion << "." << minorVersion << "." << revision << std::endl;
    std::cout << "OpenGL Version: " << glGetString(GL_VERSION) << std::endl;
    std::cout << "GLSL Version: " << glGetString(GL_SHADING_LANGUAGE_VERSION) << std::endl;

    // init OpenGL, vbo glsl after setting RC
    initGL();
    initGLSL();
    initVBO();

    // load bitmap font from array, call it after GL/GLSL is initialized
    bmFont.loadFont(fontCourier20, bitmapCourier20);

    // init GLFW callbacks
    glfwSetFramebufferSizeCallback(window, framebufferSizeCallback);
    glfwSetKeyCallback(window, keyCallback);
    glfwSetMouseButtonCallback(window, mouseButtonCallback);
    glfwSetCursorPosCallback(window, cursorPosCallback);
    glfwSetErrorCallback(errorCallback);

    // init projection matrix
    toPerspective();

    // main rendering loop
    glfwSwapInterval(1); // frequency
    while(!glfwWindowShouldClose(window))
    {
        // get frame time
        double currTime = glfwGetTime();
        double frameTime = currTime - runTime;
        runTime = currTime;

        // draw
        preFrame(frameTime);
        frame(frameTime);
        postFrame(frameTime);

        // get events
        glfwPollEvents();
    }

    // terminate program
    glfwTerminate();
    std::cout << "Terminate program" << std::endl;
    return 0;
}



///////////////////////////////////////////////////////////////////////////////
// initialize OpenGL
// disable unused features
///////////////////////////////////////////////////////////////////////////////
void initGL()
{
    glPixelStorei(GL_UNPACK_ALIGNMENT, 4);      // 4-byte pixel alignment

    // enable /disable features
    //glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);
    //glHint(GL_LINE_SMOOTH_HINT, GL_NICEST);
    //glHint(GL_POLYGON_SMOOTH_HINT, GL_NICEST);
    glEnable(GL_DEPTH_TEST);
    glEnable(GL_CULL_FACE);

    glClearColor(0, 0, 0, 0);                   // background color
    glClearStencil(0);                          // clear stencil buffer
    glClearDepth(1.0f);                         // 0 is near, 1 is far
    glDepthFunc(GL_LEQUAL);
}



///////////////////////////////////////////////////////////////////////////////
// create glsl programs
///////////////////////////////////////////////////////////////////////////////
bool initGLSL()
{
    const int MAX_LENGTH = 2048;
    char log[MAX_LENGTH];
    int logLength = 0;

    // create shader and program
    GLuint vsId = glCreateShader(GL_VERTEX_SHADER);
    GLuint fsId = glCreateShader(GL_FRAGMENT_SHADER);
    progId = glCreateProgram();

    // load shader sources
    glShaderSource(vsId, 1, &vsSource, NULL);
    glShaderSource(fsId, 1, &fsSource, NULL);

    // compile shader sources
    glCompileShader(vsId);
    glCompileShader(fsId);

    //@@ debug
    int vsStatus, fsStatus;
    glGetShaderiv(vsId, GL_COMPILE_STATUS, &vsStatus);
    if(vsStatus == GL_FALSE)
    {
        glGetShaderiv(vsId, GL_INFO_LOG_LENGTH, &logLength);
        glGetShaderInfoLog(vsId, MAX_LENGTH, &logLength, log);
        std::cout << "===== Vertex Shader Log =====\n" << log << std::endl;
    }
    glGetShaderiv(fsId, GL_COMPILE_STATUS, &fsStatus);
    if(fsStatus == GL_FALSE)
    {
        glGetShaderiv(fsId, GL_INFO_LOG_LENGTH, &logLength);
        glGetShaderInfoLog(fsId, MAX_LENGTH, &logLength, log);
        std::cout << "===== Fragment Shader Log =====\n" << log << std::endl;
    }

    // attach shaders to the program
    glAttachShader(progId, vsId);
    glAttachShader(progId, fsId);

    // link program
    glLinkProgram(progId);

    // get uniform/attrib locations
    glUseProgram(progId);
    uniformMatrixModelView           = glGetUniformLocation(progId, "matrixModelView");
    uniformMatrixModelViewProjection = glGetUniformLocation(progId, "matrixModelViewProjection");
    uniformMatrixNormal              = glGetUniformLocation(progId, "matrixNormal");
    uniformLightPosition             = glGetUniformLocation(progId, "lightPosition");
    uniformLightAmbient              = glGetUniformLocation(progId, "lightAmbient");
    uniformLightDiffuse              = glGetUniformLocation(progId, "lightDiffuse");
    uniformLightSpecular             = glGetUniformLocation(progId, "lightSpecular");
    uniformMap0                      = glGetUniformLocation(progId, "map0");
    uniformTextureUsed               = glGetUniformLocation(progId, "textureUsed");
    attribVertexPosition = glGetAttribLocation(progId, "vertexPosition");
    attribVertexNormal   = glGetAttribLocation(progId, "vertexNormal");
    attribVertexColor    = glGetAttribLocation(progId, "vertexColor");
    attribVertexTexCoord = glGetAttribLocation(progId, "vertexTexCoord");

    // set uniform values
    float lightPosition[] = {0, 0, 1, 0};
    float lightAmbient[]  = {0.3f, 0.3f, 0.3f, 1};
    float lightDiffuse[]  = {0.7f, 0.7f, 0.7f, 1};
    float lightSpecular[] = {1.0f, 1.0f, 1.0f, 1};
    glUniform4fv(uniformLightPosition, 1, lightPosition);
    glUniform4fv(uniformLightAmbient, 1, lightAmbient);
    glUniform4fv(uniformLightDiffuse, 1, lightDiffuse);
    glUniform4fv(uniformLightSpecular, 1, lightSpecular);
    glUniform1i(uniformMap0, 0);
    glUniform1i(uniformTextureUsed, 0);

    // unbind GLSL
    glUseProgram(0);
    glDeleteShader(vsId);
    glDeleteShader(fsId);

    // check GLSL status
    int linkStatus;
    glGetProgramiv(progId, GL_LINK_STATUS, &linkStatus);
    if(linkStatus == GL_FALSE)
    {
        glGetProgramiv(progId, GL_INFO_LOG_LENGTH, &logLength);
        glGetProgramInfoLog(progId, MAX_LENGTH, &logLength, log);
        std::cout << "===== GLSL Program Log =====\n" << log << std::endl;
        return false;
    }
    else
    {
        return true;
    }
}



///////////////////////////////////////////////////////////////////////////////
// copy vertex data to VBO and VA state to VAO
///////////////////////////////////////////////////////////////////////////////
void initVBO()
{
    if(!vaoDefault)
        glGenVertexArrays(1, &vaoDefault);

    // create vertex array object to store all vertex array states only once
    if(!vaoCube)
        glGenVertexArrays(1, &vaoCube);
    glBindVertexArray(vaoCube);

    // create vertex buffer objects
    if(!vboCube)
        glGenBuffers(1, &vboCube);

    // store vertex data to VBO for cube
    glBindBuffer(GL_ARRAY_BUFFER, vboCube);
    glBufferData(GL_ARRAY_BUFFER, sizeof(vertices1)+sizeof(normals1)+sizeof(colors1)+sizeof(texCoords1), 0, GL_STATIC_DRAW);
    glBufferSubData(GL_ARRAY_BUFFER, 0, sizeof(vertices1), vertices1);                                                    // copy vertices starting from 0 offest
    glBufferSubData(GL_ARRAY_BUFFER, sizeof(vertices1), sizeof(normals1), normals1);                                      // copy normals after vertices
    glBufferSubData(GL_ARRAY_BUFFER, sizeof(vertices1)+sizeof(normals1), sizeof(colors1), colors1);                       // copy colours after normals
    glBufferSubData(GL_ARRAY_BUFFER, sizeof(vertices1)+sizeof(normals1)+sizeof(colors1), sizeof(texCoords1), texCoords1); // copy tex coords after colors

    if(!iboCube)
        glGenBuffers(1, &iboCube);

    // store index data to VBO
    glGenBuffers(1, &iboCube);
    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, iboCube);
    glBufferData(GL_ELEMENT_ARRAY_BUFFER, sizeof(indices1), indices1, GL_STATIC_DRAW);

    // enable vertex array attributes for bound VAO
    glEnableVertexAttribArray(attribVertexPosition);
    glEnableVertexAttribArray(attribVertexNormal);
    glEnableVertexAttribArray(attribVertexColor);
    glEnableVertexAttribArray(attribVertexTexCoord);

    // store vertex array pointers to bound VAO
    glVertexAttribPointer(attribVertexPosition, 3, GL_FLOAT, false, 0, 0);
    glVertexAttribPointer(attribVertexNormal, 3, GL_FLOAT, false, 0, (void*)sizeof(vertices1));
    glVertexAttribPointer(attribVertexColor, 3, GL_FLOAT, false, 0, (void*)(sizeof(vertices1)+sizeof(normals1)));
    glVertexAttribPointer(attribVertexTexCoord, 2, GL_FLOAT, false, 0, (void*)(sizeof(vertices1)+sizeof(normals1)+sizeof(colors1)));

    // create vertex array object for tetrahedron
    if(!vaoTetra)
        glGenVertexArrays(1, &vaoTetra);
    glBindVertexArray(vaoTetra);
    if(!vboTetra)
        glGenBuffers(1, &vboTetra);

    // store vertex data to VBO for Tetra
    glBindBuffer(GL_ARRAY_BUFFER, vboTetra);
    glBufferData(GL_ARRAY_BUFFER, sizeof(vertices2)+sizeof(normals2)+sizeof(colors2), 0, GL_STATIC_DRAW);
    glBufferSubData(GL_ARRAY_BUFFER, 0, sizeof(vertices2), vertices2);                              // copy vertices starting from 0 offest
    glBufferSubData(GL_ARRAY_BUFFER, sizeof(vertices2), sizeof(normals2), normals2);                // copy normals after vertices
    glBufferSubData(GL_ARRAY_BUFFER, sizeof(vertices2)+sizeof(normals2), sizeof(colors2), colors2); // copy colours after normals

    // enable vertex array attributes for bound VAO
    glEnableVertexAttribArray(attribVertexPosition);
    glEnableVertexAttribArray(attribVertexNormal);
    glEnableVertexAttribArray(attribVertexColor);

    // store vertex array pointers to bound VAO
    glVertexAttribPointer(attribVertexPosition, 3, GL_FLOAT, false, 0, 0);
    glVertexAttribPointer(attribVertexNormal, 3, GL_FLOAT, false, 0, (void*)sizeof(vertices2));
    glVertexAttribPointer(attribVertexColor, 3, GL_FLOAT, false, 0, (void*)(sizeof(vertices2)+sizeof(normals2)));

    // unbind
    glBindVertexArray(vaoDefault);
    glBindBuffer(GL_ARRAY_BUFFER, 0);
    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);
}



///////////////////////////////////////////////////////////////////////////////
// initialize global variables
///////////////////////////////////////////////////////////////////////////////
bool initSharedMem()
{
    windowWidth = fbWidth = WINDOW_WIDTH;
    windowHeight = fbHeight = WINDOW_HEIGHT;

    runTime = 0;

    mouseLeftDown = mouseRightDown = mouseMiddleDown = false;
    mouseX = mouseY = 0;

    cameraAngleX = cameraAngleY = 0.0f;
    cameraDistance = CAMERA_DISTANCE;

    drawMode = 0; // 0:fill, 1: wireframe, 2:points

    vboCube = vboTetra = iboCube = vaoCube = vaoTetra = vaoDefault = texId = 0;

    return true;
}



///////////////////////////////////////////////////////////////////////////////
// clean up global vars
///////////////////////////////////////////////////////////////////////////////
void clearSharedMem()
{
    glDeleteBuffers(1, &vboCube);
    glDeleteBuffers(1, &iboCube);
    glDeleteVertexArrays(1, &vaoCube);
    glDeleteBuffers(1, &vboTetra);
    glDeleteVertexArrays(1, &vaoTetra);
    glDeleteVertexArrays(1, &vaoDefault);
    glDeleteTextures(1, &texId);
    vboCube = vboTetra = iboCube = vaoCube = vaoTetra = vaoDefault = texId = 0;
}



///////////////////////////////////////////////////////////////////////////////
// display info messages
///////////////////////////////////////////////////////////////////////////////
void showInfo()
{
    toOrtho();

    // call it once before drawing text to configure orthographic projection
    bmFont.setWindowSize(windowWidth, windowHeight);

    std::stringstream ss;
    ss << std::fixed << std::setprecision(3);

    int x = 1;
    int y = windowHeight - bmFont.getBaseline();
    bmFont.setColor(1, 1, 1, 1);

    ss << "Cube: " << sizeof(vertices1) / (3 * sizeof(float)) << " vertices" << std::ends;
    bmFont.drawText(x, y, ss.str().c_str());
    ss.str("");
    y -= bmFont.getHeight();

    ss << "Tetra: " << sizeof(vertices2) / (3 * sizeof(float)) << " vertices" << std::ends;
    bmFont.drawText(x, y, ss.str().c_str());
    ss.str("");
    y -= bmFont.getHeight();

    ss << "Press D key to change drawing mode." << std::ends;
    bmFont.drawText(1, 1, ss.str().c_str());
    ss.str("");

    // unset floating format
    ss << std::resetiosflags(std::ios_base::fixed | std::ios_base::floatfield);

    // go back to perspective mode
    toPerspective();
}



///////////////////////////////////////////////////////////////////////////////
// set projection matrix as orthogonal
///////////////////////////////////////////////////////////////////////////////
void toOrtho()
{
    const float N = -1.0f;
    const float F = 1.0f;

    // get current dimensions
    glfwGetWindowSize(window, &windowWidth, &windowHeight); // get window size
    glfwGetFramebufferSize(window, &fbWidth, &fbHeight);    // get framebuffer size

    // set viewport to be the entire framebuffer size
    glViewport(0, 0, (GLsizei)fbWidth, (GLsizei)fbHeight);

    // construct ortho projection matrix, not framebuffer size
    matrixProjection.identity();
    matrixProjection[0]  =  2.0f / windowWidth;
    matrixProjection[5]  =  2.0f / windowHeight;
    matrixProjection[10] = -2.0f / (F - N);
    matrixProjection[12] = -1.0f;
    matrixProjection[13] = -1.0f;
    matrixProjection[14] = -(F + N) / (F - N);
}



///////////////////////////////////////////////////////////////////////////////
// set the projection matrix as perspective
///////////////////////////////////////////////////////////////////////////////
void toPerspective()
{
    const float N = 0.1f;
    const float F = 100.0f;
    const float PI = acos(-1.0f);
    const float FOV_Y = 40.0f / 180.0f * PI;    // in radian

    // get current dimensions
    glfwGetWindowSize(window, &windowWidth, &windowHeight); // get window size
    glfwGetFramebufferSize(window, &fbWidth, &fbHeight);    // get framebuffer size

    // set viewport to be the entire framebuffer size
    glViewport(0, 0, (GLsizei)fbWidth, (GLsizei)fbHeight);

    // construct perspective projection matrix
    float aspectRatio = (float)(windowWidth) / windowHeight;
    float tangent = tanf(FOV_Y / 2.0f);     // tangent of half fovY
    float h = N * tangent;                  // half height of near plane
    float w = h * aspectRatio;              // half width of near plane
    matrixProjection.identity();
    matrixProjection[0]  =  N / w;
    matrixProjection[5]  =  N / h;
    matrixProjection[10] = -(F + N) / (F - N);
    matrixProjection[11] = -1;
    matrixProjection[14] = -(2 * F * N) / (F - N);
    matrixProjection[15] =  0;
}




///////////////////////////////////////////////////////////////////////////////
// render each frame
///////////////////////////////////////////////////////////////////////////////
void preFrame(double frameTime)
{
}

void frame(double frameTime)
{
    // clear buffer
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT | GL_STENCIL_BUFFER_BIT);

    Matrix4 matrixNormal;
    Matrix4 matrixModel;
    Matrix4 matrixView;
    Matrix4 matrixModelView;
    Matrix4 matrixModelViewProjection;

    // transform camera
    matrixView.rotateY(cameraAngleY);   // heading
    matrixView.rotateX(cameraAngleX);   // pitch
    matrixView.translate(0, 0, -cameraDistance);

    // bind GLSL
    glUseProgram(progId);
    //glActiveTexture(GL_TEXTURE0);
    //glBindTexture(GL_TEXTURE_2D, texId);
    glUniform1i(uniformTextureUsed, 0);

    // set matrix uniforms every frame
    matrixModel.translate(-1.5f, 0, 0);
    matrixModelView = matrixView * matrixModel;
    matrixModelViewProjection = matrixProjection * matrixModelView;
    matrixNormal = matrixModelView;
    matrixNormal.setColumn(3, Vector4(0,0,0,1));
    glUniformMatrix4fv(uniformMatrixModelView, 1, false, matrixModelView.get());
    glUniformMatrix4fv(uniformMatrixModelViewProjection, 1, false, matrixModelViewProjection.get());
    glUniformMatrix4fv(uniformMatrixNormal, 1, false, matrixNormal.get());

    // draw a cube
    glBindVertexArray(vaoCube);
    glDrawElements(GL_TRIANGLES,            // primitive type
                   36,                      // # of indices
                   GL_UNSIGNED_INT,         // data type
                   (void*)0);               // ptr to indices


    // set matrix uniforms every frame
    matrixModel.identity();
    matrixModel.translate(1.5f, 0, 0);
    matrixModelView = matrixView * matrixModel;
    matrixModelViewProjection = matrixProjection * matrixModelView;
    matrixNormal = matrixModelView;
    matrixNormal.setColumn(3, Vector4(0,0,0,1));
    glUniformMatrix4fv(uniformMatrixModelView, 1, false, matrixModelView.get());
    glUniformMatrix4fv(uniformMatrixModelViewProjection, 1, false, matrixModelViewProjection.get());
    glUniformMatrix4fv(uniformMatrixNormal, 1, false, matrixNormal.get());

    // draw a tetrahedron
    glBindVertexArray(vaoTetra);
    glDrawArrays(GL_TRIANGLES, 0, 12);

    // unbind
    //glBindTexture(GL_TEXTURE_2D, 0);
    glBindVertexArray(vaoDefault);
    glUseProgram(0);

    showInfo();

    glfwSwapBuffers(window);
}

void postFrame(double frameTime)
{
    static double elapsedTime = 0.0;
    static int frameCount = 0;
    elapsedTime += frameTime;
    ++frameCount;
    if(elapsedTime > 1.0)
    {
        double fps = frameCount / elapsedTime;
        elapsedTime = 0;
        frameCount = 0;
        //std::cout << "FPS: " << fps << std::endl;
    }
}







//=============================================================================
// GLFW CALLBACKS
//=============================================================================
void errorCallback(int error, const char* description)
{
    std::cout << "[ERROR]: " << description << std::endl;
}

void framebufferSizeCallback(GLFWwindow* window, int w, int h)
{
    toPerspective();
    std::cout << "Framebuffer resized: " << fbWidth << "x" << fbHeight
              << " (Window: " << windowWidth << "x" << windowHeight << ")" << std::endl;
}

void keyCallback(GLFWwindow* window, int key, int scancode, int action, int mods)
{
    if(key == GLFW_KEY_ESCAPE && action == GLFW_PRESS)
    {
        //std::cout << "Closing program" << std::endl;
        clearSharedMem();
        glfwSetWindowShouldClose(window, GLFW_TRUE);
    }
    else if(key == GLFW_KEY_D && action == GLFW_PRESS)
    {
        ++drawMode;
        drawMode %= 3;
        if(drawMode == 0)        // fill mode
        {
            glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
            glEnable(GL_DEPTH_TEST);
            glEnable(GL_CULL_FACE);
        }
        else if(drawMode == 1)  // wireframe mode
        {
            glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
            glDisable(GL_DEPTH_TEST);
            glDisable(GL_CULL_FACE);
        }
        else                    // point mode
        {
            glPolygonMode(GL_FRONT_AND_BACK, GL_POINT);
            glDisable(GL_DEPTH_TEST);
            glDisable(GL_CULL_FACE);
        }
    }
}

void mouseButtonCallback(GLFWwindow* window, int button, int action, int mods)
{
    // remember mouse position
    glfwGetCursorPos(window, &mouseX, &mouseY);

    if(button == GLFW_MOUSE_BUTTON_LEFT)
    {
        if(action == GLFW_PRESS)
            mouseLeftDown = true;
        else if(action == GLFW_RELEASE)
            mouseLeftDown = false;
    }
    else if(button == GLFW_MOUSE_BUTTON_RIGHT)
    {
        if(action == GLFW_PRESS)
            mouseRightDown = true;
        else if(action == GLFW_RELEASE)
            mouseRightDown = false;
    }
    else if(button == GLFW_MOUSE_BUTTON_MIDDLE)
    {
        if(action == GLFW_PRESS)
            mouseMiddleDown = true;
        else if(action == GLFW_RELEASE)
            mouseMiddleDown = false;
    }
}

void cursorPosCallback(GLFWwindow* window, double x, double y)
{
    if(mouseLeftDown)
    {
        cameraAngleY += (x - mouseX);
        cameraAngleX += (y - mouseY);
        mouseX = x;
        mouseY = y;
    }
    if(mouseRightDown)
    {
        cameraDistance -= (y - mouseY) * 0.2f;
        mouseY = y;
    }
}
